<?php
/**
 * Hooks for ArticleComments extension
 *
 * @file
 * @ingroup Extensions
 */

class ArticleCommentsHooks {

	// A counter to store the number of <comments /> elements found on the page
	private static $mCnt = 0;

	// Sets up the ArticleComments Parser hook for tags used in this extension
	public static function onParserSetup(&$parser) {

		// Whenever the <comments /> tag is encountered, execute the parserTagDump function
		$parser->setHook( 'comments', 'ArticleCommentsHooks::parserTagDump' );
		
		return true;
	}
	
	/**
	 * Parser hook handler for <comments/>
	 *
	 * @param string $data: The content of the tag.
	 * @param array $attribs: The attributes of the tag.
	 * @param Parser $parser: Parser instance available to render
	 *  wikitext into html, or parser methods.
	 * @param PPFrame $frame: Can be used to see what template
	 *  arguments ({{{1}}}) this hook was used with.
	 *
	 * @return string: HTML to insert in the page.
	 */

	public static function parserTagDump($data, $attribs, $parser, $frame) {
		global $wgArticleCommentsDefaults, $wgUser, $wgRequest;
		
		// Short circuit for anything other than action=view or action=purge
		if ($wgRequest->getVal('action') && $wgRequest->getVal('action')!='view' && $wgRequest->getVal('action')!='purge') 
			return '';
		
		self::$mCnt++;
		if ($wgArticleCommentsDefaults['disableanon']==true && !$wgUser->isLoggedIn()) {
			// Get the html for a comment form with disabled controls for displaying on the web page
			$commentForm .= ArticleCommentsHooks::generateCommentForm( $parser->mTitle, false, $attribs ); 
		}
		else
		{
			// Get the html for an enabled comment form for displaying on the web page
			$commentForm .= ArticleCommentsHooks::generateCommentForm( $parser->mTitle, true, $attribs );
		}
		
		// Hide content from the Parser using base64 encoding to avoid mangling.
		// Note: Content will be decoded after Tidy has finished its processing of the page.
		return '<pre>@ENCODED@'.base64_encode($commentForm).'@ENCODED@</pre>';
	}


 
	/**
	 * Generates and returns an ArticleComment form.
	 * @param Title $title The title of the article on which the form will appear.
	 * @param boolean $controls_enabled Whether the forms controls should be enabled or disabled
	 * @param Array $params A hash of parameters containing rendering options.
	 */
	private static function generateCommentForm( $title = null, $controls_enabled = false, $params = array() ) {
 
		global $wgUser, $wgScript, $wgArticleCommentsDefaults, $wgContentLang, $wgContLang;
		$wcl = ($wgContentLang ? $wgContentLang : $wgContLang);
		
		# Use wgTitle if title is not specified
		if ($title==null) {
			global $wgTitle;
			$title = $wgTitle;
		}
		
		$ac = 'article-comments-';
		$formAction = $wgScript.'?title='.$wcl->getNsText(NS_SPECIAL).':ProcessComment';
		
		// Build out the comment form.
		$content = "\n".'<div class="commentFormContainer">'."\n";
		
		
		$content .= '<h3>' . wfMessage($ac.'form-heading')->inContentLanguage()->text() . '</h3>';
		$content .= '<p>' . wfMessage($ac.'form-description')->inContentLanguage()->text() . '</p>';
		
		$content .= '<form id="commentForm'.self::$mCnt.'" class="commentForm" method="post" action="'.$formAction.'">'."\n";
		$content .= '<input class="commentFormInput" type="hidden" id="titleKey" name="titleKey" value="'.$title->getDBKey().'" />'."\n";
		$content .= '<input class="commentFormInput" type="hidden" id="titleNS" name="titleNS" value="'.$title->getNamespace().'" />'."\n";
		
		// Build commenter name section
		$content .= '<div id="commenterNameField'.self::$mCnt.'" class="commenterNameField">'."\n";
		$content .= '<label for="commenterName'.self::$mCnt.'" name="labelCommenterName" id="labelCommenterName">'."\n";
		$content .= wfMessage( $ac.'name-field' )->inContentLanguage()->text()."\n";
		$content .= '</label>'."\n";
		$content .= '<input class="commentFormInput" id="commenterName'.self::$mCnt.'" type="text" name="commenterName" />'."\n";
		$content .= '</div>'."\n";
		
		//wfDebug('Hello ' . $wgArticleCommentsDefaults['showurlfield']);
		
		// Build commenter url section
		if ($wgArticleCommentsDefaults['showurlfield']==true){
			$content .= '<div id="commenterURLField'.self::$mCnt.'" class="commenterURLField">'."\n";
			$content .= '<label for="commenterURL'.self::$mCnt.'" name="labelCommenterURL" id="labelCommenterURL">'."\n";
			$content .= wfMessage( $ac.'url-field' )->inContentLanguage()->text()."\n";
			$content .= '</label>'."\n";
		    	$content .= '<input class="commentFormInput" id="commenterURL'.self::$mCnt.'" type="text" name="commenterURL" />'."\n";
		    	$content .= '</div>'."\n";
		}
		
		// Build commenter town country section
		if ($wgArticleCommentsDefaults['showtowncountryfield']==true){
			$content .= '<div id="commenterTownCountryField'.self::$mCnt.'" class="commenterTownCountryField">'."\n";
			$content .= '<label for="commenterTownCountry'.self::$mCnt.'" name="labelCommenterTownCountry" id="labelCommenterTownCountry">'."\n";
			$content .= wfMessage( $ac.'town-country' )->inContentLanguage()->text()."\n";
			$content .= '</label>'."\n";
		    	$content .= '<input class="commentFormInput" id="commenterTownCountry'.self::$mCnt.'" type="text" name="commenterTownCountry" />'."\n";
		    	$content .= '</div>'."\n";
		}
		
		$content .= '<div id="commentLabel'.self::$mCnt.'">'."\n";
		$content .= '<label for="comment'.self::$mCnt.'" name="labelComment" id="labelComment">'."\n";
		$content .= wfMessage($ac.'comment-field')->inContentLanguage()->text()."\n";
		$content .= '</label>'."\n";
		$content .= '</div>'."\n";
		$content .= '<div id="commentField'.self::$mCnt.'" class="commentField">'."\n";
		$content .= '<textarea class="commentFormInput" id="comment'.self::$mCnt.'" name="comment" style="width:30em" rows="5"></textarea>'."\n";
		$content .= '</div>'."\n";
		$content .= '<div id="commentSubmitButton'.self::$mCnt.'">'."\n";
		$content .= '<input class="commentFormInput submitComment" type="submit" value="'.wfMessage($ac.'submit-button')->inContentLanguage()->text().'"'.$input_disabled.' />'."\n";
		$content .= '</div>'."\n";
		$content .= '</form>'."\n";
		$content .= '</div>'."\n";
				
		# Inline Jquery to make form behavior more rich
		$content .= "<script type='text/javascript'>\n";
		$content .= "\t$(document).ready(function() {\n";
		
		# Prefill the name and url input boxes if the user is logged in.
		if ($wgUser->isLoggedIn()){
			$content .= "\t\t$('#commenterName".self::$mCnt."').val('".$wgUser->getName()."');\n";
			$content .= "\t\t$('#commenterURL".self::$mCnt."').val('http://');\n";
		}
		
		# Hides the commentForm until the "Make a comment" link is clicked
		# Note: To disable, set $wgArticleCommentsDefaults['hideForm']=false in LocalSettings.php
		if ($wgArticleCommentsDefaults['hideform']==true){
			$content .= "\t\t$('#commentForm".self::$mCnt."').hide();\n";
			$content .= "\t\t$('#commentFormContainer".self::$mCnt."').append('<p><a id=\"hiddenFormLink".self::$mCnt."\">"
				.wfMessage($ac.'leave-comment-link')->inContentLanguage()->text()
				."</a></p>');\n";
			$content .= "\t\t$('#hiddenFormLink".self::$mCnt."').click(function(){\n";
			$content .= "\t\t\t$('#commentForm".self::$mCnt."').show();\n";
			$content .= "\t\t\t$('#hiddenFormLink".self::$mCnt."').remove();\n";
			$content .= "\t\t});\n";
		}
		
		// If controls need to be disabled then disable them by adding a "disabled" attribute to 
		// the "input" elements with jquery.
		// Also prefill comment text if it has been specified by a system message.
		if ( !$controls_enabled ){
			$content .= "\t\t$('.commentFormInput').attr('disabled', 'disabled');\n";
			$pretext = wfMessage($ac.'prefilled-comment-logged-out')->inContentLanguage()->text();		
			$content .= "\t\t$('#comment".self::$mCnt."').val('" . $pretext . "');\n";			
  				
  		}
  		else
  		{
  			$pretext = wfMessage($ac.'prefilled-comment-text')->inContentLanguage()->text();
  			$content .= "\t\t$('#comment".self::$mCnt."').val('" . $pretext . "');\n";
  			$content .= "\t\t$('#comment".self::$mCnt."').click(function(){\n";
  			$content .= "\t\t\tif ($('#comment".self::$mCnt."').val()=='".$pretext."'){\n";
			$content .= "\t\t\t\t$('#comment".self::$mCnt."').val('');\n";
			$content .= "\t\t\t}\n";
			$content .= "\t\t});\n";
			$content .= "\t\t$('#comment".self::$mCnt."').blur(function(){\n";
			$content .= "\t\t\tif ($('#comment".self::$mCnt."').val()==''){\n";
			$content .= "\t\t\t\t$('#comment".self::$mCnt."').val('" . $pretext . "');\n";
			$content .= "\t\t\t}\n";
			$content .= "\t\t});\n";
 		}
		$content .= "\t});\n";
  		$content .= "</script>\n";
		
		return $content;
	}
	
	public static function beforePageDisplay( OutputPage &$out, Skin &$skin ) {
		// Load the ResourceLoader module into the OutputPage (add the css and js to the page)
		$out->addModules( 'ext.ArticleComments' );
		// required return value or MW will fall over
		return true;
	}
	
	/**
	 * Processes HTML comments with encoded content.
	 * Usage: $wgHooks['OutputPageBeforeHTML'][] = 'wfProcessEncodedContent';
	 * @param Parser $parser : Parser instance available to render
	 *  wikitext into html, or parser methods.
	 * @param String $text Article/Output text (passed by reference)
	 * @return Boolean Always true to give other hooking methods a chance to run.
	 */
	public static function onProcessEncodedContent(&$parser, &$text) {
		
		$text = preg_replace(
	        	'/<pre>@ENCODED@([0-9a-zA-Z\\+\\/]+=*)@ENCODED@<\\/pre>/e',
	        	'base64_decode("$1")',
	        	$text
	    		);
		return true;
	}
}